/***********************************************************************************************************************
*  OpenStudio(R), Copyright (c) 2008-2023, Alliance for Sustainable Energy, LLC, and other contributors. All rights reserved.
*
*  Redistribution and use in source and binary forms, with or without modification, are permitted provided that the
*  following conditions are met:
*
*  (1) Redistributions of source code must retain the above copyright notice, this list of conditions and the following
*  disclaimer.
*
*  (2) Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following
*  disclaimer in the documentation and/or other materials provided with the distribution.
*
*  (3) Neither the name of the copyright holder nor the names of any contributors may be used to endorse or promote products
*  derived from this software without specific prior written permission from the respective party.
*
*  (4) Other than as required in clauses (1) and (2), distributions in any form of modifications or other derivative works
*  may not use the "OpenStudio" trademark, "OS", "os", or any other confusingly similar designation without specific prior
*  written permission from Alliance for Sustainable Energy, LLC.
*
*  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDER(S) AND ANY CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
*  INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
*  DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER(S), ANY CONTRIBUTORS, THE UNITED STATES GOVERNMENT, OR THE UNITED
*  STATES DEPARTMENT OF ENERGY, NOR ANY OF THEIR EMPLOYEES, BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
*  EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
*  USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
*  STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
*  ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
***********************************************************************************************************************/

#ifndef UTILITIES_IDD_ROOMAIR_NODE_FIELDENUMS_HXX
#define UTILITIES_IDD_ROOMAIR_NODE_FIELDENUMS_HXX

#include <utilities/UtilitiesAPI.hpp>
#include <utilities/core/Enum.hpp>

namespace openstudio {
/** \class RoomAir_NodeFields
 *  \brief Enumeration of RoomAir:Node's Idd and Idf non-extensible fields. 
 *  \details This enumeration provides readable, and semi-robust tags to use instead
 *  of bare unsigned integer field indices. (The enumeration value names are more likely
 *  to be valid across EnergyPlus and OpenStudio releases, and will break if they become
 *  invalid.) See the OPENSTUDIO_ENUM documentation in utilities/core/Enum.hpp. The actual macro
 *  call is:
 *
 *  \code
 *  \endcode */
#ifdef SWIG 
  OPENSTUDIO_ENUM(RoomAir_NodeFields, )
#else
class RoomAir_NodeFields: public ::EnumBase<RoomAir_NodeFields> {
 public: 
  enum domain 
  {
Name, NodeType, ZoneName, HeightofNodalControlVolumeCenter,   };
  RoomAir_NodeFields()
   : EnumBase<RoomAir_NodeFields>(Name) {} 
  RoomAir_NodeFields(const std::string &t_name) 
   : EnumBase<RoomAir_NodeFields>(t_name) {} 
  RoomAir_NodeFields(int t_value) 
   : EnumBase<RoomAir_NodeFields>(t_value) {} 
  static std::string enumName() 
  { return "RoomAir_NodeFields"; }
  domain value() const { return static_cast<domain>(EnumBase<RoomAir_NodeFields>::integer_value()); }
   private:
    friend class EnumBase<RoomAir_NodeFields>;
    typedef std::pair<std::string, int> PT;
    typedef std::vector<PT> VecType;
    static VecType buildStringVec(bool isd)
    {
      struct evalue
      {
        int value; const char *name; const char *description;
      };
      const evalue a[] =
      {
{ RoomAir_NodeFields::Name, "Name", "Name"},
{ RoomAir_NodeFields::NodeType, "NodeType", "Node Type"},
{ RoomAir_NodeFields::ZoneName, "ZoneName", "Zone Name"},
{ RoomAir_NodeFields::HeightofNodalControlVolumeCenter, "HeightofNodalControlVolumeCenter", "Height of Nodal Control Volume Center"},
        { 0,0,0 }
      };
      VecType v;
      int i = 0;
      while (!(a[i].value == 0 && a[i].name == 0 && a[i].description == 0))
      {
        if (isd)
        {
          std::string description = a[i].description; 
          if (!description.empty())
          {
            v.push_back(PT(description, a[i].value));
          }
        } else {
          v.push_back(PT(a[i].name, a[i].value));
        }
        ++i;
      }
      return v;
    }
  
  };
  inline std::ostream &operator<<(std::ostream &os, const RoomAir_NodeFields &e)
  {
    return os << e.valueName() << "(" << e.value() << ")";
  }
  typedef boost::optional<RoomAir_NodeFields> OptionalRoomAir_NodeFields ;
#endif

/** \class RoomAir_NodeExtensibleFields
 *  \brief Enumeration of RoomAir:Node's Idd and Idf extensible fields
 *  to be used in conjunction with ExtensibleIndex and ExtensibleGroup.
 *  \details This enumeration provides readable, and semi-robust tags to use instead
 *  of bare unsigned integer ExtensibleIndex.field indices. (The enumeration value names
 *  are more likely to be valid across EnergyPlus and OpenStudio releases, and will break
 *  if they become invalid.)See the OPENSTUDIO_ENUM documentation in utilities/core/Enum.hpp. The actual macro
 *  call is:
 *
 *  \code
 *  \endcode */
#ifdef SWIG 
  OPENSTUDIO_ENUM(RoomAir_NodeExtensibleFields, )
#else
class RoomAir_NodeExtensibleFields: public ::EnumBase<RoomAir_NodeExtensibleFields> {
 public: 
  enum domain 
  {
SurfaceName,   };
  RoomAir_NodeExtensibleFields()
   : EnumBase<RoomAir_NodeExtensibleFields>(SurfaceName) {} 
  RoomAir_NodeExtensibleFields(const std::string &t_name) 
   : EnumBase<RoomAir_NodeExtensibleFields>(t_name) {} 
  RoomAir_NodeExtensibleFields(int t_value) 
   : EnumBase<RoomAir_NodeExtensibleFields>(t_value) {} 
  static std::string enumName() 
  { return "RoomAir_NodeExtensibleFields"; }
  domain value() const { return static_cast<domain>(EnumBase<RoomAir_NodeExtensibleFields>::integer_value()); }
   private:
    friend class EnumBase<RoomAir_NodeExtensibleFields>;
    typedef std::pair<std::string, int> PT;
    typedef std::vector<PT> VecType;
    static VecType buildStringVec(bool isd)
    {
      struct evalue
      {
        int value; const char *name; const char *description;
      };
      const evalue a[] =
      {
{ RoomAir_NodeExtensibleFields::SurfaceName, "SurfaceName", "Surface Name"},
        { 0,0,0 }
      };
      VecType v;
      int i = 0;
      while (!(a[i].value == 0 && a[i].name == 0 && a[i].description == 0))
      {
        if (isd)
        {
          std::string description = a[i].description; 
          if (!description.empty())
          {
            v.push_back(PT(description, a[i].value));
          }
        } else {
          v.push_back(PT(a[i].name, a[i].value));
        }
        ++i;
      }
      return v;
    }
  
  };
  inline std::ostream &operator<<(std::ostream &os, const RoomAir_NodeExtensibleFields &e)
  {
    return os << e.valueName() << "(" << e.value() << ")";
  }
  typedef boost::optional<RoomAir_NodeExtensibleFields> OptionalRoomAir_NodeExtensibleFields ;
#endif

} // openstudio

#endif // UTILITIES_IDD_ROOMAIR_NODE_FIELDENUMS_HXX
